function P = convertToCanonicalForm(sysParam,cType)
% Convert the system parameters to different canonical forms  
%
% This function caclulates the change of basis matrix which can convert a given
% system parameters to different canonical forms
%
% INPUTS
%   sysParam - system parameters
%   cType    - canonical form to which we need to convert it to
%              1: RCF, 2:JCF
%
% OUTPUTS
%   P        - change of basis matrix which converts the paramters to the chosen
%              canonical form. The parameters can then be converted using the
%              function transformBasis.m
%
% EXAMPLE
% 
%
%% Written by : Avinash Ravichandran, Rizwan Chaudhry
%% $DATE      : 09-Sep-2011 17:57:43 $
%% $REVISION  : 0.5.0 $




switch cType 
    case 1 % Reachability Canonical Form
        structCheck = testStructContents(sysParam,{'A','B'});
        if ~structCheck
            error('Missing Parameters. Cannot convert to canonical form');
        end

        A  = sysParam.A;
        B  = sysParam.B;
        n  = size(A,1);
        nv = size(B,2);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % Single Input System
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (nv==1)
            O=[];
            for i=1:n
                O=[O,A^(i-1)*B];
            end
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % Multiple Input System
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (nv>1)
            idx=[];
            count=1;
            m=1;
            t=1;
            O=B(:,1);
            while (count <n)
                if (rank([O A^t*B(:,m)])~=rank(0))
                    O=[O A^t*B(:,m)];
                    idx=[idx t];
                    count=count+1;
                end
                t=t+1;
                if (t==n)
                    m=m+1;
                    t=0;
                end

            end
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % Constructing the Basis Change Matrix
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        Oinv = inv(O);
        q    = Oinv(n,:);
        P2   = [];

        for i = 1:n
            P2 = [P2;q*A^(i-1)];
        end
        P = P2;

    case 2 % Jordan Canonical Form 
        structCheck = testStructContents(sysParam,{'A','C'});
        if ~structCheck
            error('Missing Parameters. Cannot convert to canonical form');
        end

        A = sysParam.A;
        p = size(sysParam.C,1);
        n = size(A,2)
        I = eye(n);

        if p>1
            C = sum(sysParam.C,1);
        else
            C = sysParam.C;
        end

        [P,Cc,Ac] = realJordanForm(A);
        M         = kron(I,Ac)+ kron(-A',I);
        T         = kron(I,Cc);
        alt       = [M;T];
        Q         = reshape(pinv(alt)*[zeros(n^2,1);C'],n,n);
        P         = Q;

end

