function sysParam = optimalSystemID(dataMatrix,order,params)
% Identify the Linear Dynamical System parameters using N4SID
%
% INPUTS
%   dataMatrix - p x F vector matrix or r x c x F sequence matrix
%   order      - [n nv] vector, [Default: nv=1]
%   params     - parameter structure containing the various parameters
%   .Rfull     - determines if the R matrix is returned as a full matrix
%                or as a diagonal matrix [Default: 0].
%   .class     - determines the different subset of parameters that need
%                to be identified [Default: 1]
%                1: Basic parameters,
%                2: Basic + noise parameters
%                3: All parameters
%                4: All parameters for the forward innovation model
%   .Areg      - regularizes A to enforce stability (eig(A)<1) [Default: 1]
%
% OUTPUTS
%   sysParam   - output structure containing all the system parameters
%
% EXAMPLE
% 
%
%% Written by : Avinash Ravichandran, Rizwan Chaudhry
%% $DATE      : 09-Sep-2011 17:57:43 $
%% $REVISION  : 0.5.0 $

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initial Parameter Checks and Preprocessing
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
 
% Checking the Data Matrix
F = size(dataMatrix,3);

if (size(dataMatrix,3)~=1)
    I = double(reshape(dataMatrix,[],F));
else
    F = size(dataMatrix,2);
    I = dataMatrix;
end

% Checking if we have noise order 
if length(order)==1
    n    = order(1);
    nv   = 1;
else
    n    = order(1);
    nv   = order(2);
end

% Parsing Parameters
dParams.Rfull = 1;
dParams.class = 1;
dParams.Areg  = 1;
params        = testParamValidity(dParams,params);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Basic Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Creating Mean Subtracted sequence
C0         = mean(I,2);
Y          = I - repmat(C0,1,F);

% Perform N4SID
data = iddata(Y');
model = n4sid(data, n, 'InitialState', 'Estimate', 'CovarianceMatrix', 'None');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Noise Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if (params.class~=1 && params.class~=4)
    % Create Bhat - 
    tau = size(Y,2);
    [U,S,V] = svd(Y, 0);
    Xhat = S(1:n,1:n)*V(:,1:n)';                                
    Vhat = Xhat(:,2:tau) - model.A*Xhat(:,1:(tau-1));
    [Uv,Sv,Vv] = svd(Vhat,0);
    Bhat = Uv(:,1:nv)*Sv(1:nv,1:nv);
    Bhat = Bhat./sqrt(tau-1);

    Zhat = S(1:n,1:n)*V(:,1:n)';

    Rhat = var(model.NoiseVariance(:));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Assigning Output Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
sysParam.Z0   = model.x0;
sysParam.A    = model.A;
sysParam.C    = model.C;
sysParam.C0   = C0;

if (params.class~=1 && params.class~=4)
    
    sysParam.Z    = Zhat;
    
    sysParam.B    = Bhat;
    sysParam.Q    = Bhat*Bhat';    % Process Noise

    sysParam.R = Rhat;
    if params.Rfull
        sysParam.R = model.NoiseVariance;    % Measurement Noise
    end
end
    
if (params.class==3 || params.class==4)
    sysParam.K = model.K;
    sysParam.R = model.NoiseVariance;
end
