function c = ORGEN(X, y, lambda, nu, varargin)
%ORGEN Oracle guided elastic net solver.
%   C = ORGEN(X, Y, LAMBDA, NU, varargin) solves the ell_1 or elastic net 
%   optimization
% 
%   C = argmin_c LAMBDA ||c||_1 + (1-LAMBDA)/2 ||c||_2^2 + NU/2 ||Y - Xc||_2^2  (*)
% 
%   using the ORacle Guided Elastic Net (ORGEN) method described in 
% 
%   Chong You, Chun-guang Li, Daniel Robinson, Rene Vidal,
%   "Oracle Based Active Set Algorithm for Scalable Elastic Net Subspace
%   Clustering", CVPR 2016.
% 
%   The algorithm is suited for large dictionaries, i.e., the number of 
%   column in X is large. Briefly, the algorithm solves a series of smaller
%   scale problems until it finds the true support.

% Input Arguments
% X, y              -- data as above
% lambda, nu        -- parameters as above
% 'EN_solver'       -- any elastic net solver, which takes an X and y and
%                      gives the solution c to (*) above.
%     'homotopy'(default): homotopy method. See function CompSens_EN_Homotopy.m
%     'rfss'         : code from Elastic-Net Regularization: Error 
%                      estimates and Active Set Methods. Download from
%                      https://sites.google.com/site/igorcarron2/cs. See 
%                      rfss.m
%     function_handle: c = EN_solver(X, y)
% 'Nsample'         -- the maximum size of active set. It only affects the 
%                      speed of the algorithm. Recommended to be set to 
%                      around (3~10)*(sparsity of c). (default 100)
% 'init  '          -- active set initialization method. Choose the initial
%                      subset of column of X.
%     'L2'(default)  : solve the corresponding L2 problem (by setting 
%                      lambda in (*) to zero) and take #Nsample entries of
%                      the maximum absolute value.
%     'random'       : randomly pick #Nsample columns.
%     'knn'          :
% 'maxiter      '   -- max number of iterations. default 10. 
% 'outflag'         -- default true. 

%{
Example:
% Generate your X and y.
 lambda = 0.99; nu = 100;
% Usage 1 (default solver):
 c = ORGEN(A, b, lambda, nu, 'Nsample', 500, 'maxiter', 50, 'outflag', 0);
% Usage 2 (user defined solver)
 EN_solver = @(X, y) rfss( X, y, lambda / nu, (1-lambda) / nu );
 c = guidedEN(X, y, lambda, nu, 'EN_solver', EN_solver, 'Nsample', Nsample);
%}

% Copyright Chong You @ Johns Hopkins University, 2016
% chong.you1987@gmail.com

% Check input parameters
if mod(length(varargin), 2) ~= 0
   error(['' mfilename '' 'needs propertyName/propertyValue pairs']);
end
% Set default 
vararg = {'EN_solver', 'homotopy', ...
          'Nsample', 100, ...
          'init', 'L2', ...
          'maxiter', 10, ...
          'outflag', true};
% Overwrite by input
vararg = vararginParser(vararg, varargin);
% Generate variables
for pair = reshape(vararg, 2, []) % pair is {propName;propValue}
   eval([pair{1} '= pair{2};']);
end

N = size(X, 2);
D = size(X, 1);

if isa(EN_solver, 'function_handle')
    EN_solver = @(X, y, c0) EN_solver(X, y);
elseif strcmpi(EN_solver, 'rfss')
    EN_solver = @(X, y, c0) rfss( X, y, lambda / nu, (1-lambda) / nu, 'init', c0);
elseif strcmpi(EN_solver, 'homotopy')
    EN_solver = @(X, y, c0) CompSens_EN_Homotopy( X, y, lambda, nu, c0);
else
    error('EN_solver no matches\n')
end

if strcmpi(init, 'L2')
%     c0 = X' / (eye(D)/nu+X*X') * y;
    c0 = X' * ( (eye(D)/nu+X*X') \ y );
    [~, ord] = sort(abs(c0), 'descend');
    supp = ord(1:Nsample);
elseif strcmpi(init, 'random')
    supp = randperm(N, Nsample);
elseif strcmpi(init, 'knn')
    [~, ord] = sort( abs(X'*y), 'descend' );
    supp = ord(1:Nsample);
end

supp = supp(:);
cs = zeros(length(supp), 1);

for iter = 1:maxiter
    % S1: Xs -> oracle v
    Xs = X(:, supp);

    cs = EN_solver(Xs, y, cs);
    delta = nu * (y - Xs*cs); % oracle 

    % S2: guided neighborhood.
    coherence = abs(X' * delta);
    coherence(supp) = 0;
    addedsupp = find(coherence > lambda); % a mask for C that out of mask is zero.
    if isempty(addedsupp) % converged
        break;
    end
    activesupp = supp(cs ~= 0);
    
    if length(activesupp) >= Nsample
        fprintf('Warning: Nsample is too small and is increased by 10 percent\n');
        Nsample = min( [ceil(Nsample * 1.1), N] );
    end
    
    if length(addedsupp) + length(activesupp) > Nsample % active set is too large
        [~, ord] = sort(coherence(addedsupp), 'descend');
        addedsupp = addedsupp(ord(1:Nsample - length(activesupp)));      
    end
    supp = [activesupp; addedsupp];
    
    if outflag
        obj = lambda * sum(abs(cs)) + (1 - lambda)/2 * sum(cs.^2) + nu/2 * sum((y - Xs*cs) .^2);
        fprintf('Iteration %d: obj: %f, new support size: %d\n', iter, obj, length(supp))
    end
    
    cs = [cs(cs ~= 0); zeros(length(addedsupp), 1)]; % serve as initial value for the next iteration
end

c = zeros(N, 1);
c(supp) = cs;


    
