%   This code tests the performance of R-graph for outlier detection on the 
%   Coil-100 image database. The code generates results in 
%   Table 3 of the paper
%
%   Chong You, Daniel Robinson, Rene Vidal,
%   "Provable Self-Representation Based Outlier Detection in a Union of 
%   Subspaces", CVPR 2017.

% Copyright Chong You @ Johns Hopkins University, 2017
% chong.you1987@gmail.com

clear all;close all;

addpath('data')
addpath('tools')
load('COIL100.mat');
% Coil-100: we use the data provided at http://www.cad.zju.edu.cn/home/dengcai/Data/MLData.html
%% Set param
Num_Inlier_Group = 4; % number of inlier groups (taken to be 1, 4 and 7 in the paper)
outlier_perc = 0.25; % percentage of outliers (taken to be 50%, 25% and 15% in the paper)
Nexperiment = 50;

fprintf('Coil100: Num_Inlier_Group = %d, outlier perc = %f, Nexperiment = %d\n', Num_Inlier_Group, outlier_perc, Nexperiment);

flag_ROC = false; % set true to generate ROC curve plots for each experiment 
%(set Nexperiment to be small if flag_ROC is set to be true!)

%% Experiment
result = zeros(3, Nexperiment);
for iexperiment = 1:Nexperiment
    subjectIdx = randperm(100, Num_Inlier_Group);
    
% Prepare data  
    % inliers are images of certain category(s) specified by subjectIdx
    datapointIdx = find(ismember(gnd, subjectIdx));
    data_inlier = fea(datapointIdx, :)';
    Ninlier = length(datapointIdx);

    % outliers are images of other categories, with at most one image taken
    % from each category.
    Noutlier = round(outlier_perc / (1-outlier_perc) * Ninlier); 
    outsubjectIdx = setdiff(1:100, subjectIdx);
    outsubjectIdx = outsubjectIdx(randperm(100 - length(subjectIdx), Noutlier));
    outpointIdx = zeros(1, Noutlier);
    for ii = 1:Noutlier
        datapointIdx = find(ismember(gnd, outsubjectIdx(ii)));
        outpointIdx(ii) = datapointIdx(randperm(length(datapointIdx), 1)); 
    end
    data_outlier = fea(outpointIdx, :)';

    % compose test data
    data = [data_inlier, data_outlier];
    data = double(data);
    s = [zeros(1, Ninlier), ones(1, Noutlier)];
    N = Ninlier + Noutlier;

% R-graph outlier detection
    tic;
    
    data = dimReduction(data, min(size(data)));
    % step 1: compute representation R from data (line 1 of Alg. 1)
    lambda = 0.95;
    alpha = 10;
    gamma = @(X, y, lambda, alpha)  alpha*lambda/max(abs(X'*y));
    EN_solver =  @(X, y) rfss( full(X), full(y), lambda / gamma(X, y, lambda, alpha), (1-lambda) / gamma(X, y, lambda, alpha) );
    R = selfRepresentation(cnormalize(data), EN_solver);
    % step 2: compute transition P from R (line 2 of Alg. 1)
    P = cnormalize(abs(R), 1)';
    % step 3: compute \pi from P (line 3 - 7 of Alg. 1)
    T = 1000;
    pi = ones(1, N) / N;
    pi_bar = zeros(1, N);
    for ii = 1:T
        pi = pi * P;
        pi_bar = pi_bar + pi;
    end
    pi_bar = pi_bar / T;
    %
    feat = - pi_bar; % larger values in feat indicate higher "outlierness"
      
% Evaluation
    time = toc;
    
    [FPR, TPR, T, AUC] = perfcurve(s, feat, 1);
    [PREC, RECA] = perfcurve(s, feat, 1, 'XCrit', 'prec', 'YCrit', 'reca');
    F1 = max(2 * (PREC .* RECA) ./ (PREC + RECA));
    fprintf('Experiment %d: AUC = %f, F1 = %f, time = %f\n', iexperiment, AUC, F1, time);

    result(:, iexperiment) = [AUC, F1, time]';
    if flag_ROC
        figure;
        plot(FPR, TPR, '-r');
        xlabel('False positive rate')
        ylabel('True positive rate')
    end
end
fprintf('Mean AUROC: %f, Mean F1: %f, Mean time: %f sec.\n', mean(result(1, :)), mean(result(2, :)), mean(result(3, :)));




    





    
